# -*- coding: utf-8 -*-
#
#*******************************************************************************
#
#  Copyright 2022 RIEGL Laser Measurement Systems
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
#  SPDX-License-Identifier: Apache-2.0
#
#*******************************************************************************
#
"""
Database open settings
"""

from ctypes import byref, c_void_p, c_uint64, c_char_p

from . import library
from . import utilities


class OpenSettings:
    """
    Database open settings

    This class defines settings for opening a point cloud database.
    """

    def __init__(self, context):
        """
        Default constructor

        All values are set to default values.
        """
        self.context = context
        self.handle = c_void_p(None)
        self.context.check(
            library.handle.rdb_pointcloud_open_settings_new(
                self.context.handle, byref(self.handle)
            )
        )

    def __del__(self):
        """Destroy settings object"""
        if self.handle != c_void_p(None):
            library.handle.rdb_pointcloud_open_settings_delete(
                self.context.handle, byref(self.handle)
            )
            self.handle = c_void_p(None)

    def load(self, json):
        """
        Load settings from JSON string

        This function parses the given JSON string and applies all available
        properties - missing properties are silently ignored (i.e. the value
        remains unchanged). When parsing the JSON string fails, an exception
        is thrown.

        Example JSON string:

            {
                "cache_size": 524288000
            }
        """
        buffer = utilities.to_rdb_string(json)
        self.context.check(
            library.handle.rdb_pointcloud_open_settings_json_load(
                self.context.handle, self.handle, buffer
            )
        )

    def save(self):
        """
        Save settings to JSON string
        See load()
        """
        buffer = c_char_p(None)
        self.context.check(
            library.handle.rdb_pointcloud_open_settings_json_save(
                self.context.handle, self.handle, byref(buffer)
            )
        )
        return utilities.to_std_string(buffer)

    @property
    def cache_size(self):
        """
        Point cache size

        The database engine may buffer read and write operations in an
        internal cache. This value defines the cache size in bytes (octets).

        Default: 500 MB
        """
        buffer = c_uint64(0)
        self.context.check(
            library.handle.rdb_pointcloud_open_settings_get_cache_size_64(
                self.context.handle, self.handle, byref(buffer)
            )
        )
        return buffer.value

    @cache_size.setter
    def cache_size(self, value):
        buffer = c_uint64(value)
        self.context.check(
            library.handle.rdb_pointcloud_open_settings_set_cache_size_64(
                self.context.handle, self.handle, buffer
            )
        )
